// ダークテーマ処理専用モジュール
class DarkThemeManager {
  constructor() {
    // 基本色マッピング（拡張版）
    this.colorMap = {
      // 基本色
      "rgb(241, 241, 241)": "rgb(35, 35, 35)",    // #f1f1f1 → #232323
      "rgb(34, 34, 34)": "rgb(242, 242, 242)",    // #222 → #f2f2f2
      "rgb(255, 255, 255)": "rgb(35, 35, 35)",    // 白 → #232323
      "rgb(0, 0, 0)": "rgb(242, 242, 242)",       // 黒 → #f2f2f2
      "rgb(96, 96, 96)": "rgb(242, 242, 242)",    // rgb(96, 96, 96) → #f2f2f2
      "rgb(0, 0, 102)": "rgb(0, 119, 211)",       // rgb(0, 0, 102) → rgb(0, 119, 211)
      "rgb(85, 85, 85)": "rgb(200, 200, 200)",    // rgb(85, 85, 85) → rgb(200, 200, 200)
      "rgb(246, 246, 246)": "rgb(35, 35, 35)",    // rgb(246, 246, 246) → rgb(35, 35, 35)
      "rgb(51, 51, 51)": "rgb(200, 200, 200)",    // rgb(51, 51, 51) → rgb(200, 200, 200)
      "rgb(233, 233, 233)": "rgb(65, 65, 65)",    // rgb(233, 233, 233) → rgb(65, 65, 65)
      "rgb(130, 130, 130)": "rgb(200, 200, 200)", // rgb(130, 130, 130) → rgb(200, 200, 200)
      "rgb(255, 250, 188)": "rgb(63, 63, 63)",
      "rgb(234, 248, 253)": "rgb(63, 63, 63)",
      "rgb(255, 249, 235)": "rgb(63, 63, 63)",
      
      // 追加の色マッピング
      "rgb(248, 248, 248)": "rgb(40, 40, 40)",    // 薄いグレー
      "rgb(245, 245, 245)": "rgb(42, 42, 42)",    // 薄いグレー
      "rgb(240, 240, 240)": "rgb(45, 45, 45)",    // 薄いグレー
      "rgb(238, 238, 238)": "rgb(47, 47, 47)",    // 薄いグレー
      "rgb(229, 229, 229)": "rgb(55, 55, 55)",    // 薄いグレー
      "rgb(220, 220, 220)": "rgb(60, 60, 60)",    // 薄いグレー
      "rgb(204, 204, 204)": "rgb(80, 80, 80)",    // 中間グレー
      "rgb(170, 170, 170)": "rgb(120, 120, 120)", // 中間グレー
      "rgb(153, 153, 153)": "rgb(140, 140, 140)", // 中間グレー
      "rgb(136, 136, 136)": "rgb(160, 160, 160)", // 中間グレー
      "rgb(119, 119, 119)": "rgb(180, 180, 180)", // 中間グレー
      "rgb(102, 102, 102)": "rgb(200, 200, 200)", // 中間グレー
      "rgb(68, 68, 68)": "rgb(220, 220, 220)",    // 濃いグレー
      "rgb(17, 17, 17)": "rgb(240, 240, 240)",    // 濃いグレー
      
      // ボタン・リンク色
      "rgb(0, 123, 255)": "rgb(0, 119, 211)",     // 青系
      "rgb(40, 167, 69)": "rgb(40, 167, 69)",     // 緑系（そのまま）
      "rgb(220, 53, 69)": "rgb(220, 53, 69)",     // 赤系（そのまま）
      "rgb(255, 193, 7)": "rgb(255, 193, 7)",     // 黄系（そのまま）
      "rgb(23, 162, 184)": "rgb(23, 162, 184)",   // シアン系（そのまま）
      
      // 入力フィールド色
      "rgb(255, 255, 255)": "rgb(35, 35, 35)",    // 入力フィールド背景
      "rgb(73, 80, 87)": "rgb(200, 200, 200)",    // 入力フィールドテキスト
      "rgb(206, 212, 218)": "rgb(100, 100, 100)", // 入力フィールドボーダー
    };

    this.shadowMap = {
      "rgba(0, 0, 0, 0.2) 0px 0px 2px 0px": "rgb(255 255 255 / 20%) 0px 0px 2px 0px",
      "rgba(0, 0, 0, 0.1) 0px 1px 1px 0px": "rgb(255 255 255 / 10%) 0px 1px 1px 0px",
      "rgba(0, 0, 0, 0.15) 0px 1px 3px 0px": "rgb(255 255 255 / 15%) 0px 1px 3px 0px",
      "rgba(0, 0, 0, 0.12) 0px 2px 4px 0px": "rgb(255 255 255 / 12%) 0px 2px 4px 0px",
      "rgba(0, 0, 0, 0.25) 0px 4px 8px 0px": "rgb(255 255 255 / 25%) 0px 4px 8px 0px",
      "rgba(0, 0, 0, 0.3) 0px 8px 16px 0px": "rgb(255 255 255 / 30%) 0px 8px 16px 0px"
    };

    // 処理済み要素のキャッシュ
    this.processedElements = new WeakSet();
    
    // MutationObserverの設定
    this.mutationObserver = null;
    this.observerInitialized = false;
    
    // パフォーマンス統計
    this.stats = {
      totalElements: 0,
      processedElements: 0,
      skippedElements: 0,
      lastUpdate: Date.now()
    };
  }

  // メインの適用関数
  applyDarkMode() {
    // 初回適用時のみ全体を処理
    this.updateStyles(this.colorMap, this.shadowMap);
    
    // MutationObserverを設定（まだ設定されていない場合）
    if (!this.observerInitialized) {
      this.initializeMutationObserver();
    }
    
    // 統計情報を更新
    this.updateStats();
  }

  // MutationObserverの初期化
  initializeMutationObserver() {
    if (this.mutationObserver) {
      this.mutationObserver.disconnect();
    }

    this.mutationObserver = new MutationObserver((mutations) => {
      let shouldUpdate = false;
      
      mutations.forEach((mutation) => {
        if (mutation.type === 'childList') {
          // 新しい要素が追加された場合
          mutation.addedNodes.forEach((node) => {
            if (node.nodeType === Node.ELEMENT_NODE) {
              this.processNewElement(node);
              shouldUpdate = true;
            }
          });
        } else if (mutation.type === 'attributes') {
          // 属性が変更された場合（スタイル変更など）
          if (mutation.attributeName === 'style' || mutation.attributeName === 'class') {
            this.processElement(mutation.target);
            shouldUpdate = true;
          }
        }
      });

      if (shouldUpdate) {
        this.updateStats();
      }
    });

    // 観察を開始
    this.mutationObserver.observe(document.body, {
      childList: true,
      subtree: true,
      attributes: true,
      attributeFilter: ['style', 'class']
    });

    this.observerInitialized = true;
  }

  // 新しい要素の処理
  processNewElement(element) {
    // 単一要素の処理
    this.processElement(element);
    
    // 子要素も処理
    const children = element.querySelectorAll('*');
    children.forEach(child => this.processElement(child));
  }

  // スタイルを一括更新（パフォーマンス改善版）
  updateStyles(colorMap, shadowMap) {
    // メインドキュメントのスタイルを更新
    this.applyStylesOptimized(document, colorMap, shadowMap);

    // iframe内のスタイルを更新
    this.processIframes(document, colorMap, shadowMap);
  }

  // iframe処理を最適化
  processIframes(doc, colorMap, shadowMap) {
    const iframes = doc.querySelectorAll('iframe');
    if (iframes.length === 0) return;

    iframes.forEach(iframe => {
      try {
        const iframeDocument = iframe.contentDocument || iframe.contentWindow.document;
        if (iframeDocument) {
          this.applyStylesOptimized(iframeDocument, colorMap, shadowMap);
          // 再帰的にiframe内のiframeも処理
          this.processIframes(iframeDocument, colorMap, shadowMap);
        }
      } catch (e) {
        // クロスオリジンiframeへのアクセスエラーを無視（ログ出力しない）
      }
    });
  }

  // 最適化されたスタイル適用
  applyStylesOptimized(doc, colorMap, shadowMap) {
    // バッチ処理で効率化
    const elements = doc.querySelectorAll('*');
    const elementsToProcess = Array.from(elements).filter(element => {
      // 処理済みの要素はスキップ
      if (this.processedElements.has(element)) {
        this.stats.skippedElements++;
        return false;
      }
      
      // 特定のクラスを持つ要素はスキップ
      if (element.classList.contains('sc-aXZVg') && element.classList.contains('kLYaYr')) {
        return false;
      }
      
      return true;
    });

    // バッチ処理
    this.processBatch(elementsToProcess, colorMap, shadowMap);
  }

  // バッチ処理で効率化
  processBatch(elements, colorMap, shadowMap) {
    const batchSize = 100; // 一度に処理する要素数
    let index = 0;

    const processBatch = () => {
      const batch = elements.slice(index, index + batchSize);
      batch.forEach(element => {
        this.processElement(element, colorMap, shadowMap);
        this.processedElements.add(element);
        this.stats.processedElements++;
      });

      index += batchSize;
      if (index < elements.length) {
        // 非同期で次のバッチを処理（UIブロックを防ぐ）
        requestAnimationFrame(processBatch);
      }
    };

    processBatch();
  }

  // 単一要素の処理（色マップ指定可能版）
  processElement(element, colorMap = this.colorMap, shadowMap = this.shadowMap) {
    try {
      const computedStyle = getComputedStyle(element);
      
      // 要素タイプ別の特別処理
      this.applyElementSpecificStyles(element, computedStyle, colorMap, shadowMap);
      
      // 基本的な色プロパティの処理
      this.applyBasicColorStyles(element, computedStyle, colorMap);
      
      // ボックスシャドウの処理
      this.applyBoxShadowStyles(element, computedStyle, shadowMap);
      
      // CSS変数の処理
      this.applyCSSVariables(element);
      
      this.stats.totalElements++;
    } catch (e) {
      // 要素処理エラーは無視（ログ出力しない）
    }
  }

  // 要素タイプ別の特別処理
  applyElementSpecificStyles(element, computedStyle, colorMap, shadowMap) {
    const tagName = element.tagName.toLowerCase();
    
    switch (tagName) {
      case 'input':
        this.applyInputStyles(element, computedStyle, colorMap);
        break;
      case 'button':
        this.applyButtonStyles(element, computedStyle, colorMap);
        break;
      case 'select':
        this.applySelectStyles(element, computedStyle, colorMap);
        break;
      case 'textarea':
        this.applyTextareaStyles(element, computedStyle, colorMap);
        break;
      case 'a':
        this.applyLinkStyles(element, computedStyle, colorMap);
        break;
      case 'table':
      case 'th':
      case 'td':
        this.applyTableStyles(element, computedStyle, colorMap);
        break;
    }
  }

  // 基本的な色プロパティの処理
  applyBasicColorStyles(element, computedStyle, colorMap) {
    const colorProps = [
      'backgroundColor', 'color', 'borderColor', 'borderTopColor', 
      'borderRightColor', 'borderBottomColor', 'borderLeftColor',
      'outlineColor', 'textDecorationColor', 'caretColor'
    ];

    colorProps.forEach(prop => {
      const currentColor = computedStyle[prop];
      if (currentColor && colorMap[currentColor]) {
        element.style[prop] = colorMap[currentColor];
      }
    });
  }

  // ボックスシャドウの処理
  applyBoxShadowStyles(element, computedStyle, shadowMap) {
    const currentShadow = computedStyle.boxShadow;
    if (currentShadow && currentShadow !== 'none') {
      Object.keys(shadowMap).forEach(oldShadow => {
        if (currentShadow.includes(oldShadow)) {
          element.style.boxShadow = currentShadow.replace(oldShadow, shadowMap[oldShadow]);
        }
      });
    }
  }

  // CSS変数の処理
  applyCSSVariables(element) {
    const cssVariables = {
      '--bs-body-bg': 'rgb(35, 35, 35)',
      '--bs-body-color': 'rgb(242, 242, 242)',
      '--bs-border-color': 'rgb(100, 100, 100)',
      '--bs-light': 'rgb(65, 65, 65)',
      '--bs-dark': 'rgb(200, 200, 200)',
      '--bs-secondary': 'rgb(160, 160, 160)',
      '--bs-success': 'rgb(40, 167, 69)',
      '--bs-danger': 'rgb(220, 53, 69)',
      '--bs-warning': 'rgb(255, 193, 7)',
      '--bs-info': 'rgb(23, 162, 184)',
      '--bs-primary': 'rgb(0, 119, 211)'
    };

    Object.keys(cssVariables).forEach(variable => {
      element.style.setProperty(variable, cssVariables[variable]);
    });
  }

  // 入力フィールドの特別処理
  applyInputStyles(element, computedStyle, colorMap) {
    element.style.backgroundColor = 'rgb(45, 45, 45)';
    element.style.color = 'rgb(242, 242, 242)';
    element.style.borderColor = 'rgb(100, 100, 100)';
  }

  // ボタンの特別処理
  applyButtonStyles(element, computedStyle, colorMap) {
    const bgColor = computedStyle.backgroundColor;
    
    // プライマリボタン
    if (bgColor === 'rgb(0, 123, 255)') {
      element.style.backgroundColor = 'rgb(0, 119, 211)';
      element.style.color = 'rgb(255, 255, 255)';
    }
    // セカンダリボタン
    else if (bgColor === 'rgb(108, 117, 125)') {
      element.style.backgroundColor = 'rgb(160, 160, 160)';
      element.style.color = 'rgb(255, 255, 255)';
    }
    // 通常のボタン
    else {
      element.style.backgroundColor = 'rgb(65, 65, 65)';
      element.style.color = 'rgb(242, 242, 242)';
      element.style.borderColor = 'rgb(100, 100, 100)';
    }
  }

  // セレクトボックスの特別処理
  applySelectStyles(element, computedStyle, colorMap) {
    element.style.backgroundColor = 'rgb(45, 45, 45)';
    element.style.color = 'rgb(242, 242, 242)';
    element.style.borderColor = 'rgb(100, 100, 100)';
  }

  // テキストエリアの特別処理
  applyTextareaStyles(element, computedStyle, colorMap) {
    element.style.backgroundColor = 'rgb(45, 45, 45)';
    element.style.color = 'rgb(242, 242, 242)';
    element.style.borderColor = 'rgb(100, 100, 100)';
  }

  // リンクの特別処理
  applyLinkStyles(element, computedStyle, colorMap) {
    element.style.color = 'rgb(0, 119, 211)';
    
    // 訪問済みリンク
    if (element.matches(':visited')) {
      element.style.color = 'rgb(123, 104, 238)';
    }
    
    // ホバー時の色
    element.addEventListener('mouseenter', () => {
      element.style.color = 'rgb(40, 167, 211)';
    });
    
    element.addEventListener('mouseleave', () => {
      element.style.color = 'rgb(0, 119, 211)';
    });
  }

  // テーブルの特別処理
  applyTableStyles(element, computedStyle, colorMap) {
    element.style.backgroundColor = 'rgb(45, 45, 45)';
    element.style.color = 'rgb(242, 242, 242)';
    element.style.borderColor = 'rgb(100, 100, 100)';
    
    // 交互の行色
    if (element.tagName.toLowerCase() === 'tr') {
      const index = Array.from(element.parentNode.children).indexOf(element);
      if (index % 2 === 0) {
        element.style.backgroundColor = 'rgb(50, 50, 50)';
      } else {
        element.style.backgroundColor = 'rgb(40, 40, 40)';
      }
    }
  }

  // 統計情報の更新
  updateStats() {
    this.stats.lastUpdate = Date.now();
    
    // デバッグ出力は削除
  }

  // パフォーマンス情報の取得
  getPerformanceInfo() {
    return {
      ...this.stats,
      cacheSize: this.processedElements.size || 0,
      observerActive: this.observerInitialized,
      colorMapSize: Object.keys(this.colorMap).length,
      shadowMapSize: Object.keys(this.shadowMap).length
    };
  }

  // キャッシュクリア
  clearCache() {
    this.processedElements = new WeakSet();
    this.stats = {
      totalElements: 0,
      processedElements: 0,
      skippedElements: 0,
      lastUpdate: Date.now()
    };
  }

  // ダークテーマの無効化
  disable() {
    if (this.mutationObserver) {
      this.mutationObserver.disconnect();
      this.mutationObserver = null;
    }
    this.observerInitialized = false;
    this.clearCache();
  }

  // 色の自動検出と追加
  autoDetectColors() {
    const detectedColors = new Set();
    const elements = document.querySelectorAll('*');
    
    elements.forEach(element => {
      const computedStyle = getComputedStyle(element);
      const colorProps = ['backgroundColor', 'color', 'borderColor'];
      
      colorProps.forEach(prop => {
        const color = computedStyle[prop];
        if (color && color !== 'rgba(0, 0, 0, 0)' && color !== 'transparent') {
          detectedColors.add(color);
        }
      });
    });

    // 検出された色を返す
    return Array.from(detectedColors).filter(color => !this.colorMap[color]);
  }

  // 色マッピングを追加する関数
  addColorMapping(originalColor, darkColor) {
    this.colorMap[originalColor] = darkColor;
  }

  // シャドウマッピングを追加する関数
  addShadowMapping(originalShadow, darkShadow) {
    this.shadowMap[originalShadow] = darkShadow;
  }

  // 色マッピングを取得する関数
  getColorMap() {
    return { ...this.colorMap };
  }

  // シャドウマッピングを取得する関数
  getShadowMap() {
    return { ...this.shadowMap };
  }

  // 色の一括追加
  addColorMappings(colorMappings) {
    Object.assign(this.colorMap, colorMappings);
  }

  // シャドウの一括追加
  addShadowMappings(shadowMappings) {
    Object.assign(this.shadowMap, shadowMappings);
  }
}

// グローバルに公開（content.jsで使用するため）
window.DarkThemeManager = DarkThemeManager;
